C      *****************************************************************
C      * Subroutine Decomp                                             *
C      * Note that the variables names in this routine are those used  *
C      * by GEM so that this routine will be more readily understood   *
C      * when referring to the GEM documentation.                      *
C      *                                                               *
C      * Variables:                                                    *
C      * AC        -Input - C to N ratio of cellulose.  Units are      *
C      *                    g C/g N.                                   *
C      * AE        -Input - C to N ratio of extractives.  Units are    *
C      *                    g C/g N.                                   *
C      * AH        -Input - C to N ratio of humus.  Units are g C/g N. *
C      * AL        -Input - C to N ratio of lignin.  Units are g C/g N.*
C      * Area      -Input - The area of the current layer. Units are   *
C      *                    m**2.                                      *
C      * bCE       -Input - Cellulose to extractives transformation    *
C      *                    efficiency.  Units are                     *
C      *                    g extractives C/g cellulose C.             *
C      * bEL       -Input - Extractives to lignin transformation       *
C      *                    efficiency. Units are                      *
C      *                    g lignin C/g extractives C.                *
C      * bHE       -Input - Humus to extractives transformation        *
C      *                    efficiency.  Units are                     *
C      *                    g extractives C/g humus C.                 *
C      * bLE       -Input - Lignin to extractives transformation       *
C      *                    efficiency.  Units are                     *
C      *                    g extractives C/g lignin C.                *
C      * bLH       -Input - Lignin to humus transformation             *
C      *                    efficiency.  Units are                     *
C      *                    g humus C/g lignin C.                      *
C      * CC        -Input - Cellulose C.  Units are g C/m**2.          *
C      * dCE       -Input - Cellulose to extractives transformation    *
C      *                    rate parameter.  Units are 1/timestep.     *
C      * DcmpDetail-Input - Indicates whether detailed output for      *
C      *                    the decomposition routine should be output *
C      *                    (TRUE) or not (FALSE).  Unitless.          *
C      * DcmpOut   -Input - The output file number for detailed output *
C      *                    of decomposition routine.  Unitless.       *
C      * dEL       -Input - Extractives to lignin transformation       *
C      *                    rate parameter.  Units are 1/timestep.     *
C      * dHE       -Input - Humus to extractives transformation        *
C      *                    rate parameter.  Units are 1/timestep.     *
C      * dLE       -Input - Lignin to extractives transformation       *
C      *                    rate parameter.  Units are 1/timestep.     *
C      * dLH       -Input - Lignin to humus transformation rate        *
C      *                    parameter.  Units are 1/timestep.          *
C      * EC        -Input - Extractives C.  Units are g C/m**2.        *
C      * EN        -Input - Extractives N.  Units are g N/m**2.        *
C      * HC        -Input - Humus C.  Units are g C/m**2.              *
C      * ILC       -Local - Ligno-cellulose index.  Unitless.          *
C      * KND       -Input - Decomposition N half saturation constant.  *
C      *                    Units are g N/m**2.                        *
C      * LC        -Input - Lignin C.  Units are g C/m**2.             *
C      * Lyr       -Input - The current layer's number (for output).   *
C      *                    Unitless.                                  *
C      * M         -Input - Moisture.  Units are mm available water.   *
C      * MFcn      -Local - Soil moisture response function for        *
C      *                    decomposition.  Unitless.                  *
C      * MO        -Input - Decomposition optimum soil moisture.       *
C      *                    Units are mm available water.              *
C      * Ms        -Input - Decomposition soil moisture breadth        *
C      *                    parameter.  Unitless.                      *
C      * N         -Input - The soil inorganic nitrogen.  Units are    *
C      *                    (g N/m**2).                                *
C      * NetNMin   -Local - The net N mineralization.  Units are       *
C      *                    gN/m**2/timestep.                          *
C      * NICE      -Output- N immobilization associated with TCEC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NIEL      -Output- N immobilization associated with TELC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NIHE      -Output- N immobilization associated with THEC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NILE      -Output- N immobilization associated with TLEC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NILH      -Output- N immobilization associated with TLHC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NMCE      -Output- N mineralization associated with TCEC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NMEL      -Output- N mineralization associated with TELC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NMHE      -Output- N mineralization associated with THEC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NMLE      -Output- N mineralization associated with TLEC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NMLH      -Output- N mineralization associated with TLHC.     *
C      *                    Units are g N/m**2/timestep.               *
C      * NRatio    -Local - The ratio N / (N + KND).  Unitless.        *
C      * qd        -Input - Curvature paramter for QDT.  Units are     *
C      *                    1/deg C.                                   *
C      * QDT       -Local - Temperature response function for          *
C      *                    decomposition.  Unitless.                  *
C      * REL       -Local - The respiration associated with TELC.      *
C      *                    Units are g C/m**2/timestep.               *
C      * RCE       -Local - The respiration associated with TCEC.      *
C      *                    Units are g C/m**2/timestep.               *
C      * RHE       -Local - The respiration associated with THEC.      *
C      *                    Units are g C/m**2/timestep.               *
C      * RLE       -Local - The respiration associated with TLEC.      *
C      *                    Units are g C/m**2/timestep.               *
C      * RLH       -Local - The respiration associated with TLHC.      *
C      *                    Units are g C/m**2/timestep.               *
C      * S         -Input - Decomposition soil moisture skewness       *
C      *                    parameter.  Unitless.                      *
C      * SC        -Input - A matrix, of size NDepSp by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each species.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * SE        -Input - ILC shielding parameter for extractives.   *
C      *                    Unitless.                                  *
C      * T         -Input - Temperature.  Units are deg C.             *
C      * TandM     -Local - The product of the soil moisture response  *
C      *                    function and the temperature response      *
C      *                    function.  Unitless.                       *
C      * TCEC      -Output- Transformation of C from cellulose to      *
C      *                    extractives. Units are g C/m**2/timestep.  *
C      * TELC      -Output- Transformation of C from extractives to    *
C      *                    lignin.  Units are g C/m**2/timestep.      *
C      * Temp1..3  -Local - Variables used to hold sub-calculations    *
C      *                    for equations.  Unitless.                  *
C      * THEC      -Output- Transformation of C from humus to          *
C      *                    extractives.  Units are g C/m**2/timestep. *
C      * TLEC      -Output- Transformation of C from lignin to         *
C      *                    extractives. Units are g C/m**2/timestep.  *
C      * TLHC      -Output- Transformation of C from lignin to         *
C      *                    humus.  Units are g C/m**2/timestep.       *
C      * TMD       -Input - Maximum temperature for decomposition.     *
C      *                    Units are deg C.                           *
C      * Tab       -Local - The tab character.  Unitless.              *
C      * TOD       -Input - Optimum temperature for decomposition.     *
C      *                    Units are deg C.                           *
C      * TotResp   -Output- The sum of the respirations associated with*
C      *                    each transformation.  Units are            *
C      *                    g C/m**2/timestep.                         *
C      * V         -Local - Decomposition soil moisture exponent.      *
C      *                    Unitless.                                  *
C      *****************************************************************
       SUBROUTINE DECOMP(AC,     AE,   AL,   AH,   
     >                   BCE,    BEL,  BHE,  BLE,    
     >                   BLH,    CC,   DCE,  DEL,  
     >                   DHE,    DLE,  DLH,  EC,   
     >                   EN,     HC,   KND,  LC, 
     >                   M,      MO,   MS,   N,    
     >                   QD,     S,    SC,   SE,  
     >                   T,      TMD,  TOD,   DCMPDETAIL,
     >                   DCMPOUT,LYR,  AREA,
     >                   NICE,   NIEL, NIHE, NILE, 
     >                   NILH,   NMCE, NMEL, NMHE, 
     >                   NMLE,   NMLH, TCEC, TELC,  
     >                   THEC,   TLEC, TLHC, RCE,  
     >                   REL,    RHE,  RLE,  RLH,
     >                   TOTRESP)

							IMPLICIT NONE
       REAL*8 AC, AE, AH, AL, AREA, BCE, BEL, BHE, BLE 
       REAL*8 BLH, CC, DCE, DEL, DHE, DLE, DLH, EC, EN, HC
       REAL*8 ILC, KND, LC, M, MFCN, MO, MS, N, NETNMIN
       REAL*8 NICE, NIEL, NIHE, NILE, NILH
       REAL*8 NMCE, NMEL, NMHE, NMLE, NMLH, NRATIO, QD, QDT
       REAL*8 RCE, REL, RHE, RLE, RLH
       REAL*8 S, SC, SE, T, TANDM, TCEC, TELC, TEMP1, TEMP2
       REAL*8 TEMP3, THEC, TLEC, TLHC, TMD, TOD,  TOTRESP, V
       CHARACTER*1 TAB
       LOGICAL DCMPDETAIL
       INTEGER DCMPOUT, LYR

C      *------------------------*
C      * Set the tab character. *
C      *------------------------*
       TAB = CHAR(9)

C      *-------------------------------------*
C      * Calculate the ligno-cellulose index *
C      *-------------------------------------*
       ILC = LC / (LC + CC)

C      *------------------------------------------------------------*
C      * Calculate the ratio of the amount of nitrogen to the sum   *
C      * of the amount of nitrogen and its half saturation constant *
C      * for use in simplifying later equations.                    *
C      *------------------------------------------------------------*
       NRATIO = N / (N + KND)

C      *----------------------------------------------------*
C      * Calculate the decomposition soil moisture exponent *
C      *----------------------------------------------------*
       TEMP1 = MO ** S
       V = ((M ** S - TEMP1) / (TEMP1 - 100 ** S)) ** 2

C      *---------------------------------------------------*
C      * Calculate the soil moisture response function for *
C      * decomposition.                                    *
C      *---------------------------------------------------*
       MFCN = MS ** V

C      *-------------------------------------------------*
C      * Calculate the temperature response function for *
C      * decomposition.                                  *
C      *-------------------------------------------------*
       IF (T .LE. TMD) THEN
          TEMP1 = EXP(QD * (T - TOD)) 
          TEMP2 = ((TMD - T) / (TMD - TOD))
          TEMP3 = QD * (TMD - TOD)
          QDT = TEMP1 * TEMP2 ** TEMP3
       ELSE
          QDT = 0.0
       ENDIF

C      *------------------------------------------------------------*
C      * Calculate the product of the temperature response function *
C      * and the moisture response function for simplification of  *
C      * later equations.                                           *
C      *------------------------------------------------------------*
       TANDM = MFCN * QDT

C      *-----------------------------------------------------*
C      * Calculate the transformation of C from cellulose to *
C      * extractives.                                        *
C      *-----------------------------------------------------*
       TEMP1 = EXP(-SC * ILC)
       TCEC = BCE * DCE * TANDM * NRATIO * TEMP1 * CC

C      *-------------------------------------------------*
C      * Calculate the respiration associated with TCEC. *
C      *-------------------------------------------------*
       RCE = (1/BCE - 1) * TCEC

C      *-------------------------------------------------------*
C      * Calculate the transformation of C from extractives to *
C      * lignin.                                               *
C      *-------------------------------------------------------*
       TEMP1 = EXP(-SE * ILC)
       IF ((EC / EN * BEL).GE. AL) THEN
          TELC = BEL * DEL * TANDM * NRATIO * TEMP1 * EC
       ELSE
          TELC =  BEL * DEL * TANDM * TEMP1 * EC
       ENDIF

C      *-------------------------------------------------*
C      * Calculate the respiration associated with TELC. *
C      *-------------------------------------------------*
       REL = (1/BEL - 1) * TELC

C      *--------------------------------------------------*
C      * Calculate the transformation of C from lignin to *
C      * extractives.                                     *
C      *--------------------------------------------------*
       TLEC = BLE * DLE * TANDM * LC

C      *-------------------------------------------------*
C      * Calculate the respiration associated with TLEC. *
C      *-------------------------------------------------*
       RLE = (1/BLE - 1) * TLEC

C      *-------------------------------------------------*
C      * Calculate the transformation of C from humus to *
C      * extractives.                                    *
C      *-------------------------------------------------*
       THEC = BHE * DHE * TANDM * HC

C      *-------------------------------------------------*
C      * Calculate the respiration associated with THEC. *
C      *-------------------------------------------------*
       RHE = (1/BHE - 1) * THEC

C      *--------------------------------------------------*
C      * Calculate the transformation of C from lignin to *
C      * humus.                                           *
C      *--------------------------------------------------*
       TLHC = BLH * DLH * TANDM * NRATIO * LC

C      *-------------------------------------------------*
C      * Calculate the respiration associated with TLHC. *
C      *-------------------------------------------------*
       RLH = (1/BLH - 1) * TLHC

C      *------------------------------------*
C      * Add together all the respirations. *
C      *------------------------------------*
       TOTRESP = RCE + REL + RLE + RHE + RLH 

C      *-----------------------------------------------------*
C      * Calculate the mineralization and the immobilization *
C      * associated with TCEC.                               * 
C      *-----------------------------------------------------*
       NMCE = TCEC / (BCE * AC)
       NICE = TCEC / AE

C      *-----------------------------------------------------*
C      * Calculate the mineralization and the immobilization *
C      * associated with TELC. Note that the input ratio     *
C      * AE cannot be used in the denominator because the C  *
C      * to N ratio is not a constant when material is       *
C      * COMING OUT of the extractives.  This is the only    *
C      * case when this happens.                             * 
C      *-----------------------------------------------------*
       NMEL = (TELC * EN) / (BEL * EC)
       NIEL = TELC / AL

C      *-----------------------------------------------------*
C      * Calculate the mineralization and the immobilization *
C      * associated with TLEC.                               * 
C      *-----------------------------------------------------*
       NMLE = TLEC / (BLE * AL)
       NILE = TLEC / AE

C      *-----------------------------------------------------*
C      * Calculate the mineralization and the immobilization *
C      * associated with THEC.                               * 
C      *-----------------------------------------------------*
       NMHE = THEC / (BHE * AH)
       NIHE = THEC / AE

C      *-----------------------------------------------------*
C      * Calculate the mineralization and the immobilization *
C      * associated with TLHC.                               * 
C      *-----------------------------------------------------*
       NMLH = TLHC / (BLH * AL)
       NILH = TLHC / AH

C      *-------------------------------------*
C      * Calculate the net N mineralization. *
C      *-------------------------------------*
       NETNMIN = NMCE + NMEL + NMLE + NMLH + NMHE - 
     >           NICE - NIEL - NILE - NILH - NIHE

C      *----------------------------------------------------*
C      * If detailed output was chosen, output information. *
C      *----------------------------------------------------*
       IF (DCMPDETAIL) THEN
          WRITE(DCMPOUT, 101) LYR,TAB, TOTRESP*AREA,TAB, 
     >    NETNMIN*AREA,TAB, EC*AREA,TAB, CC*AREA,TAB, 
     >    LC*AREA,TAB, HC*AREA,TAB, EN*AREA,TAB, N*AREA, TAB 
  101     FORMAT(' ',1(I2,A1),25(E12.6,A1))
       ENDIF

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
        
